/**
* @(#) ViewerPanel.java   

* modifications made by Aaron N. Chang, PhD 
* 11/2004
* Dept of Microbiology
* Univ of Washington, Seattle
*/

/*
 * @(#)GraphInteraction.java 
 * based on the Graph.java by Sun Microsystems, Inc.
 * modified by Ralf Mrowka, Humboldt-University-Berlin, Germany, 2000
 */

/*
 * @(#)Graph.java	1.9 98/10/28
 *
 * Copyright (c) 1997, 1998 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */


package org.compbio.bioverse.viewer;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Panel;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;


public class ViewerPanel extends Panel implements Runnable, MouseListener, MouseMotionListener{
	
	// array limit parameters - affects size of the applet
	private static final int MAXEDGES = 20000;
	private static final int MAXGOCATS = 100000;
	private static final int MAXNODES = 10000;
  
  	// arrays used for data 
	public GoInfo[] GOcats = new GoInfo[MAXGOCATS];  				// # Nodes x 10 GO cats each
	public Interaction[] Interactions = new Interaction[MAXEDGES]; // array of Edges
	public Node[] Nodes = new Node[MAXNODES];						 // array of Nodes

	// note: legacy code access instance variables directly (todo: change to getter/setter)
	public int neighbours_level_actual;
	public int nGOcats;
	public int nInteractions;
	private int nNodes; 
	long cycle_number = 1 ;
	private static int tally = 0;
	
	private boolean pickfixed;
	private boolean random;
	boolean hide_relax;
	boolean freeze;
	boolean show_fu_text;
	
	private double sumium_old;
	private double max_old;

	private final Color fixedColor = Color.green.brighter();  //selected node
	private final Color MemberColor = Color.darkGray; // default node color
	private final Color nextNColor = Color.yellow.brighter(); //selected node's neighbors
	private final Color selectColor = Color.pink;
	private final Color selectedColor = Color.white;

	ViewerApplet graph;
	Graphics offgraphics;
	Image offscreen;
	Dimension offscreensize;
	Node pick;
	Thread relaxer;
	Running ru;


	/** CONSTRUCTOR
	 * 
	 * @param graph
	 */
	public ViewerPanel(ViewerApplet graph) {
		this.graph = graph;
		addMouseListener(this);
		ru = new Running();
	}

	
	/**
	 * 
	 * @return
	 */
	public int getNodeCount() {
		return nNodes;
	}
	
	/** create a new GoInfo object, store bioverseId, and and store object into array
	 * 
	 * @param bioverseId
	 * @return
	 */
	public int addGoInfoToArray(String bioverseId) {
		GoInfo n = new GoInfo();
		n.bioverseId = bioverseId; // set bioverse identifier for current entry
		GOcats[nGOcats] = n; // GOcats is the goInfo array

		return nGOcats++;
	}

	/** add a go cat to gocat array (up to MAXGOCATS)
	 * 
	 * @param bvRecord
	 * @param goCategory
	 */
	public void addGoCat(String bvRecord, String goCategory) {
		int index = findGoInfoIndex(bvRecord);  // get index for a bioverse record
	
		// find open slots in function name array to put data in
		for (int i = 0; i < GOcats[index].goList.length; i++) {
			if (GOcats[index].goList[i] == null) {			
				GOcats[index].goList[i] = goCategory;
				break;			
			}
		}
	} 


	/**mini func name that gets displayed next to a member node if so desire - defunct in AC's version
	 * 
	 * @param nodeName
	 * @param bvRecord
	 */
	public void addGoFlagToNode(String nodeName, String bvRecord) { 
		// locate or add indexes
		int n = findMember(nodeName); // node label
		int f = findGoInfoIndex(bvRecord);  // bioverse number
	
		Nodes[n].funkflag[f] = true;  
		
		if (Nodes[n].fu_lbl==""){
			Nodes[n].fu_lbl= bvRecord; // if func name has not be assigned yet, give it fu
		}else{
			Nodes[n].fu_lbl= Nodes[n].fu_lbl +"," + bvRecord; // if more than one func, separate by comma
		}
		
	} 

	/** INTERACTION HANDLING METHODS 
	 * 
	 * @param from
	 * @param to
	 * @param length
	 * @param strength
	 */
	public void addInteraction(String from, String to, int length, double strength) {
		Interaction interaction = new Interaction();
		interaction.from = findMember(from);  //converts string to an index where node is stored in Nodes[]
		interaction.to = findMember(to);
		interaction.len = length;
		interaction.strength = strength; 
	
		Interactions[nInteractions++] = interaction;
	}

	/** add node
	 * 
	 * @param label
	 * @return
	 */
	public int addNode(String label) {
		Node node = new Node();
		Dimension d = ViewerApplet.selfRef.getSize();
       
		node.x = 10+(d.width-20)*Math.random();
		node.y = 10+(d.height-20)*Math.random();
		node.lbl = label;
		node.fu_lbl = "";
		Nodes[nNodes] = node;
		node.on = true ;
		node.neighbours_level=1;
	
		return nNodes++;
	}

	/** FUNCTION LABEL METHODS*****************************
	 * 
	 * @param bioverseId
	 * @return
	 */
	public int findGoInfoIndex(String bioverseId) {
		for (int i = 0 ; i < nGOcats; i++) {
			if (GOcats[i].bioverseId.equals(bioverseId)) {   // if the label is found, return bioverseId stored index
			return i;
			}
		}
		return addGoInfoToArray(bioverseId); // if label is NOT found, add it in and return new index
	}

	/** MEMBER HANDLING METHODS*************************************
	 * 
	 * @param label
	 * @return
	 */
	public int findMember(String label) {
		for (int i = 0 ; i < nNodes ; i++) {
			if (Nodes[i].lbl.equals(label)) {
				return i;
			}
		}
		return addNode(label);
	}

	/** get a node's list of a GO cats
	 * 
	 * @param nodeLabel
	 * @return
	 */
	public String[] getGOcats (String nodeLabel) {
		int index = findMember(nodeLabel);
		int flagIndex = 0;
		Node current = Nodes[index];
		String[] temp = null;
	
		// get index of first flag set to true (note: this will be fixed later)
		for (int i = 0; i < current.funkflag.length; i++) {
			if (current.funkflag[i] == true) {
				flagIndex = i;
				break;
			}
		}
	
		temp = GOcats[flagIndex].goList;
		return temp;
	}

	/**
	 * get interactions
	 * @return
	 */
	public Interaction[] getInteractions() {
		return Interactions;
	}

	/**
	 * get nodes
	 * @return
	 */
	public Node[] getNodes() {
		return Nodes;
	}

	/**
	 * 
	 */
	public void mouseClicked(MouseEvent e) {
		int clickcount = e.getClickCount(); 

		if  (clickcount == 2) {  //double-clicked selected member will only keep those connected to it
			for (int i = 0 ; i < nNodes ; i++) {
				Node n = Nodes[i];
			
				if (!(n.nextneighbour || n.selected ) ) { //turn off all the others - removes them from the display
					n.on = false ;
				}  
			}
		}

		if  (clickcount == 3 ) { }
    
		if(!hide_relax){repaint();}
		e.consume();
	}

	/**
	 * 
	 */
	public void mouseDragged(MouseEvent e) {
		try {
			pick.x = e.getX();
			pick.y = e.getY();
			if(!hide_relax){repaint();}
			e.consume();
		} catch (NullPointerException ex) {
			// do nothing
		}
	}

	/**
	 * 
	 */
	public void mouseEntered(MouseEvent e) {
		repaint();    
	}

	// unimplemented methods
	public void mouseExited(MouseEvent e) {}
	public void mouseMoved(MouseEvent e) {}
  
  	/**
  	 * 
  	 */
	public void mousePressed(MouseEvent e) {
		addMouseMotionListener(this);
		double bestdist = Double.MAX_VALUE;
		int x = e.getX();
		int y = e.getY();
	
		try {
			for (int i = 0 ; i < nNodes ; i++) {
				Node node = Nodes[i];
					if(node.on){
						double dist = (node.x - x) * (node.x - x) + (node.y - y) * (node.y - y);
						if (dist < bestdist) {
							pick = node;   // SET PICKED node: ACTUAL OR CLOSEST TO POINT OF MOUSE CLICK
							bestdist = dist;
						}
					}
			}
		
			if(!freeze){
				pick.x = x;
				pick.y = y;
		
				// DELETE ALL NEXT NEIGHBOURS 
				for (int i = 0 ; i < nNodes ; i++) {
					Node node = Nodes[i];
					node.neighbours_level=nInteractions;
					node.nextneighbour = false;  
					node.selected = false;  
				}
		
				pick.neighbours_level=0;
				pick.selected =true ;
			
		
				// CALCULATE NEAREST NEIGHBOURS 
				for(int lev = 0 ; lev < neighbours_level_actual ; lev++){
					for (int jj = 0 ; jj < nNodes ; jj++) {
						Node aru = Nodes[jj];
						if(aru.neighbours_level==lev){ 
							for (int i = 0 ; i < nInteractions ; i++) {
								Interaction ed = Interactions[i];
							
								if (Nodes[ed.from] == aru){
									Nodes[ed.to].nextneighbour = true ;
									if(Nodes[ed.to].neighbours_level > lev+1){
									Nodes[ed.to].neighbours_level = lev+1;
									}
									Nodes[ed.to].on = true ;
									//Nodes[ed.from].selected = true ;
									Nodes[ed.from].on = true ;   
								}
							
								if (Nodes[ed.to] == aru){
									Nodes[ed.from].nextneighbour = true ;
									
									if(Nodes[ed.from].neighbours_level > lev+1){
										Nodes[ed.from].neighbours_level = lev+1;
									}
							   
									Nodes[ed.to].on = true ;
									//Nodes[ed.to].selected = true ;
									Nodes[ed.from].on = true ;    
								}
							}
						}
					}
				}
			}
		} catch (NullPointerException ex){
				// do nothing 
		}
	
		if(!hide_relax){repaint();}
			e.consume();
	} 


	/**
	 * 
	 */
	public void mouseReleased(MouseEvent e) {
		removeMouseMotionListener(this);
		if (pick != null) {
			pick.x = e.getX();
			pick.y = e.getY();
			pick.fixed = pickfixed;
			pick = null;
		}

		if(!hide_relax){
			repaint();
		}
	
		e.consume();
	}

	/** method that determines node display
	 * 
	 * @param g
	 * @param n
	 * @param fm
	 */
	public void paintMember(Graphics g, Node n, FontMetrics fm) {
		if (n.on) { // check if node should be displayed
			int x = (int)n.x;
			int y = (int)n.y;
			int w = fm.stringWidth(n.lbl) + 6;
			int h = fm.getHeight() + 2;
			int cornerWidth = 10;
			int cornerHeight = 10;
		
			g.setColor(MemberColor);
		
			 // sets font and box border color; check what type of node is being painted
			if(n.nextneighbour){ 
				g.setColor(nextNColor);
				g.fillRoundRect(x - w/2, y - h / 2, w, h, cornerWidth, cornerHeight);

				g.setColor(Color.black);
				g.drawString(n.lbl, x - (w-6)/2, (y - (h-4)/2) + fm.getAscent());
			} else if (n==pick){
				g.setColor(selectedColor);
				g.fillRoundRect(x - w/2, y - h / 2, w, h, cornerWidth, cornerHeight);
		
				g.setColor(Color.black);
				g.drawString(n.lbl, x - (w-6)/2, (y - (h-4)/2) + fm.getAscent());
			} else {
				g.fillRoundRect(x - w/2, y - h / 2, w, h, cornerWidth, cornerHeight);
				g.setColor(Color.white); 
		
				g.drawRoundRect(x - w/2, y - h / 2, w-1, h-1, cornerWidth, cornerHeight);
				g.drawString(n.lbl, x - (w-6)/2, (y - (h-4)/2) + fm.getAscent());
			}
		
			if(n.neighbours_level==0){
				g.setColor(fixedColor);
				g.fillRoundRect(x - w/2, y - h / 2, w, h, cornerWidth, cornerHeight);
		
				g.setColor(Color.black);
				g.drawString(n.lbl, x - (w-6)/2, (y - (h-4)/2) + fm.getAscent());
			}
		    
			if(show_fu_text){
				g.drawString(n.fu_lbl, x +w/2+2  , (y - (h-4)/2) + fm.getAscent());
			}
		}
	}

	
	/** relaxation algorithm : determines the "springyness" of edge connections
	 * 
	 *
	 */
	public synchronized void relax() {
		if(freeze==false){
			for (int i = 0 ; i < nInteractions ; i++) {
				Interaction e = Interactions[i];
			
				if(Nodes[e.from].on && Nodes[e.to].on){   
					double vx = Nodes[e.to].x - Nodes[e.from].x;
					double vy = Nodes[e.to].y - Nodes[e.from].y;
					double len = Math.sqrt(vx * vx + vy * vy);
			    
					len = (len == 0) ? .0001 : len;
					{ 			    
					double f = (Interactions[i].len - len) / (len * 5);
					double dx = f * vx;
					double dy = f * vy;
				
					Nodes[e.to].dx += dx;
					Nodes[e.to].dy += dy;
					Nodes[e.from].dx += -dx;
					Nodes[e.from].dy += -dy;
					}
				}
			}
	    
			for (int i = 0 ; i < nNodes ; i++) {
				Node n1 = Nodes[i];
				double dx = 0;
				double dy = 0;
			
				for (int j = 0 ; j < nNodes ; j++) {
					if (i == j) {
						continue;
					}
			    
					Node n2 = Nodes[j];
			    
					if (!(n1.on && n2.on)) {
						continue;
					}
			    
					double vx = n1.x - n2.x;
					double vy = n1.y - n2.y;
					double len = vx * vx + vy * vy;
			    
					if (len == 0) {
					//	dx += Math.random();
					// dy += Math.random();
					} else if (len < 10000) {
						dx += vx / len;
						dy += vy / len;
					}
				}
			
				double dlen = dx * dx + dy * dy;
				if (dlen > 0) {
					dlen = Math.sqrt(dlen) / 2;
					n1.dx += dx / dlen;
							n1.dy += dy / dlen;
				}
			}
	    
	    
			Dimension d = getSize();
			double sumium =0;
			double max_sumium=0;
	    
			if (cycle_number <1000000) {
				cycle_number++;
			}
	    
			for (int i = 0 ; i < nNodes ; i++) {
				Node n = Nodes[i];
		
				if (!n.fixed) {
					n.x += Math.max(-5, Math.min(5, n.dx));
					n.y += Math.max(-5, Math.min(5, n.dy));
				}
			
				if (n.x < 29) {
					n.x = 29;
				} else if (n.x > (d.width-29)) {
					n.x = d.width-29;
				}
			
				if (n.y < 16) {
					n.y = 16;
				} else if (n.y > (d.height-16)) {
					n.y = d.height-16;
				}
		
				n.dx /= 2;
				n.dy /= 2;
			
				if(hide_relax==true){
					if(Math.abs(n.dx) >  max_sumium){
						max_sumium=Math.abs(n.dx);
					}
			    
					if(Math.abs(n.dy) > max_sumium){
						max_sumium=Math.abs(n.dy);
					}
			    
					sumium += Math.abs(n.dx);
					sumium += Math.abs(n.dy);
				}
			}
	    
			if(cycle_number>20){
				if(hide_relax==true){
					if(max_sumium<1){
						ViewerApplet.selfRef.hide_relax.setState(false);
						hide_relax=false;
					}
					
					if(3<0){
						if(((Math.abs(sumium_old-sumium)/sumium) < 0.00011)){
							ViewerApplet.selfRef.hide_relax.setState(false);
							hide_relax=false;
						}
					} 
				}
			}

			sumium_old=sumium;
			max_old=max_sumium;
	    
			if(!hide_relax){
				repaint();
			}	
		}    
	} //end of relax()

	
	//
	public void run() {
		Thread me = Thread.currentThread();
    
		while (relaxer == me) {
			relax();
	   
			try {
				Thread.sleep(40);  // this sets the speed (intervals) of relaxation
			} catch (InterruptedException e) {
				break;
			}
		}
	}

	/**
	 * 
	 * @param newInteractions
	 */
	public void setInteractions(Interaction[] newInteractions) {
		Interactions = newInteractions;
	}

	/**** Member and Interaction setting/getting - for external class access from TableApplet
	 * 
	 * @param newNodes
	 */
	public void setNodes(Node[] newNodes){
		Nodes = newNodes;
	}

	/**
	 * 
	 *
	 */
	public void start() {
		relaxer = new Thread(this);
		relaxer.start();
	}

	/**
	 * 
	 *
	 */
	public void stop() {
		relaxer = null;
	}

	/**freshing the panel graphics
	 * 
	 */
	public synchronized void update(Graphics g) {
		Dimension d = getSize();

		if ((offscreen == null) || (d.width != offscreensize.width) || (d.height != offscreensize.height)) {
			offscreen = createImage(d.width, d.height);
			offscreensize = d;
			offgraphics = offscreen.getGraphics();
			offgraphics.setFont(new Font("ARIAL", Font.BOLD, 12));
		}

		offgraphics.setColor(Color.black);  //sets viewer background color
		offgraphics.fillRect(0, 0, d.width, d.height); // panel
	
		// draw interaction (edges)
		for (int i = 0 ; i < nInteractions ; i++) {
			Interaction interaction = Interactions[i];
	   
			if(Nodes[interaction.from].on && Nodes[interaction.to].on){
				// get coordinates for the edges
				int x1 = (int)Nodes[interaction.from].x;
				int y1 = (int)Nodes[interaction.from].y;
				int x2 = (int)Nodes[interaction.to].x;
				int y2 = (int)Nodes[interaction.to].y;
				int len = (int)Math.abs(Math.sqrt((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2)) - interaction.len);
				//	offgraphics.setColor((len < 10) ? arcColor1 : (len < 20 ? arcColor2 : arcColor3)) ;
		
				// edge display colors
				double strength = interaction.getStrength();
			
				if (strength >= 0.0 && strength < 0.25) {
					offgraphics.setColor(Color.red); 
				} else if (strength >= 0.25 && strength < 0.5) {
					offgraphics.setColor(Color.orange); 
				} else if (strength >= 0.5 && strength < 0.75) {
					offgraphics.setColor(Color.yellow); 
				} else if (strength >= 0.75 && strength <= 1.0) {
					offgraphics.setColor(Color.green); 
				} else {
					offgraphics.setColor(Color.darkGray); 
				}
			
				offgraphics.drawLine(x1, y1, x2, y2); // edge draw
			}
		}

		FontMetrics fm = offgraphics.getFontMetrics();
	
		// display the nodes
		for (int i = 0 ; i < nNodes ; i++) {
			paintMember(offgraphics, Nodes[i], fm);
		}
	
		g.drawImage(offscreen, 0, 0, null);
	}

}
